import asyncio
import json
import time
import traceback
import ffmpeg
import streamlink
from streamlink.session import Streamlink

# 1. 初始化 Streamlink 会话
session = Streamlink()

# 2. 设置插件选项 (兼容 Streamlink 7.0.0+ 的新写法)
# Twitch 插件参数：禁用广告、开启低延迟
session.set_option("twitch-disable-ads", True)
session.set_option("twitch-low-latency", True)
# 如果需要设置全局代理，可以取消下面注释
# session.set_option("http-proxy", "http://127.0.0.1:7890")

class LiveRelay:
    def __init__(self, config: dict, item: dict):
        self.index = item['index']
        self.id = item['id']
        self.platform = item['platform'].lower()
        self.name = item.get('name', self.id)
        self.flag = f'[{self.platform.upper()}][{self.name}]'
        self.rtmp_server = config['RTMPServer']
        # 用于记录当前是否正在推流，防止重复启动
        self.is_running = False

    async def get_stream_url(self):
        """
        异步获取直播流地址
        """
        url_map = {
            "twitch": f"https://www.twitch.tv/{self.id}",
            "youtube": f"https://www.youtube.com/channel/{self.id}/live"
        }
        target_url = url_map.get(self.platform)
        if not target_url:
            return None

        try:
            # session.streams 是阻塞操作，放入线程池中执行以防卡死异步循环
            streams = await asyncio.to_thread(session.streams, target_url)
            if streams and 'best' in streams:
                return streams['best'].url
        except Exception as e:
            # 过滤掉常见的“流不存在”错误，减少日志噪音
            if "No streams found" not in str(e):
                print(f"{self.flag} 检测直播状态时出错: {e}")
        return None

    async def run_ffmpeg(self, stream_url):
        """
        启动 FFmpeg 进行转推
        """
        self.is_running = True
        print(f"{self.flag} 正在启动推流 -> {self.rtmp_server}")
        
        try:
            # 使用 -vcodec copy 和 -acodec copy 极大地降低 CPU 占用
            process = (
                ffmpeg
                .input(stream_url)
                .output(
                    self.rtmp_server,
                    vcodec='copy',
                    acodec='copy',
                    f='flv',
                    listen=0, # 确保不是作为服务器监听
                    loglevel='error'
                )
                .run_async(pipe_stdin=True)
            )

            # 等待进程结束
            while process.poll() is None:
                await asyncio.sleep(5)
            
            print(f"{self.flag} 推流进程已结束。")
        except Exception as e:
            print(f"{self.flag} FFmpeg 运行异常: {e}")
        finally:
            self.is_running = False

    async def monitor_loop(self):
        """
        独立的监控循环
        """
        print(f"{self.flag} 监控启动...")
        while True:
            try:
                if not self.is_running:
                    stream_url = await self.get_stream_url()
                    if stream_url:
                        await self.run_ffmpeg(stream_url)
                
                # 每 30 秒轮询一次是否开播
                await asyncio.sleep(30)
            except Exception as e:
                print(f"{self.flag} 监控循环捕获到异常: {e}")
                await asyncio.sleep(10)

async def main():
    CONFIG_FILE = "config.json"
    try:
        with open(CONFIG_FILE, 'r', encoding='utf-8') as f:
            config = json.load(f)
    except FileNotFoundError:
        print(f"[ERROR] 找不到配置文件 {CONFIG_FILE}")
        return

    tasks = []
    for item in config["streams"]:
        relay = LiveRelay(config, item)
        tasks.append(asyncio.create_task(relay.monitor_loop()))
        # 错开启动时间，避免瞬间高并发请求
        await asyncio.sleep(2)

    print(f"[INFO] 已成功加载 {len(tasks)} 个监控任务。")
    await asyncio.gather(*tasks)

if __name__ == "__main__":
    while True:
        try:
            print(f"[INFO] 程序启动... 时间: {time.strftime('%Y-%m-%d %H:%M:%S')}")
            asyncio.run(main())
        except KeyboardInterrupt:
            print("[INFO] 程序由用户手动停止。")
            break
        except Exception as e:
            print(f"[CRITICAL] 主程序崩溃: {e}")
            traceback.print_exc()
            print("[INFO] 15 秒后尝试重启...")
            time.sleep(15)